import * as state from './state.js';
import * as ui from './ui.js';
import * as ipc from './ipc.js';
import * as input from './input.js';
import * as utils from './utils.js';

async function appLogic() {
    ui.fetchNotificationData();
    
    const initialData = await ipc.getInitialData();
    if (!initialData) {
        throw new Error("Failed to receive initial data from main process.");
    }
    state.setInitialData(initialData);

    const loadedSettings = await ipc.getSettings();
    state.setInitialSettings(loadedSettings);
    
    if (state.appSettings.region === 'Europe') {
        document.querySelector('link[href="style.css"]').href = 'style_eur.css';
    }

    ui.setCssAndImageSources(state.UI_URLS);
    await ipc.loadTranslations(state.appSettings.language);
    utils.applyTranslationsToDOM();

    // --- Corrected BGM Loading Logic ---
    if (state.appSettings.bgmPath === "off") {
        const offIndex = state.availableBgmTracks.findIndex(track => track.name === 'Music Off');
        state.appSettings.currentBgmIndex = offIndex !== -1 ? offIndex : 0;
        ui.bgmNameDisplay.textContent = utils.t('audiomenu_bgm_off');
    } else {
        let trackToPlay = null;
        if (state.appSettings.bgmPath === "default") {
            trackToPlay = state.availableBgmTracks.find(track => track.name === 'Default');
        } else {
            trackToPlay = state.availableBgmTracks.find(track => track.path === state.appSettings.bgmPath);
        }

        if (trackToPlay && trackToPlay.path) {
            const savedIndex = state.availableBgmTracks.findIndex(track => track.path === trackToPlay.path);
            state.appSettings.currentBgmIndex = savedIndex !== -1 ? savedIndex : 1;
            ui.bgmNameDisplay.textContent = trackToPlay.name === 'Default' ? utils.t('audiomenu_bgm_default') : trackToPlay.name;
            
            utils.backgroundMusic.src = `file:///${trackToPlay.path.replace(/\\/g, '/')}`;
            utils.backgroundMusic.play().catch(e => console.error("BGM failed to play:", e));
        } else {
            // Fallback if the saved song is missing, default to off.
            const offIndex = state.availableBgmTracks.findIndex(track => track.name === 'Music Off');
            state.appSettings.currentBgmIndex = offIndex !== -1 ? offIndex : 0;
            ui.bgmNameDisplay.textContent = utils.t('audiomenu_bgm_off');
        }
    }
    
    utils.backgroundMusic.loop = true;
    utils.backgroundMusic.volume = state.appSettings.bgmVolumeLevel / 10 * 0.5;
    utils.navigateSound.src = state.UI_URLS.navigateSoundUrl; utils.navigateSound.volume = 0.7;
    utils.confirmSound.src = state.UI_URLS.confirmSoundUrl;
    utils.cancelSound.src = state.UI_URLS.cancelSoundUrl;

    await ui.renderUI();
    ui.renderAlphaBar();
    ui.startFullscreenDisplay.textContent = state.appSettings.startFullscreen ? utils.t('common_yes') : utils.t('common_no');

    if (state.games.length > 0) {
        const initialIndex = state.appSettings.lastGameIndex || 0;
        state.setSelectedIndex(Math.max(0, Math.min(state.games.length - 1, initialIndex)));
    } else {
        ui.gameTitleElement.textContent = utils.t('game_title_no_games');
    }
    
    // Delay layout-dependent updates to ensure styles are fully applied first
    setTimeout(() => {
        ui.updateMainContentPadding();
        if (state.games.length > 0) {
            ui.updateSelection(state.selectedIndex, false);
        }
    }, 50);

    window.addEventListener('resize', () => { ui.updateMainContentPadding(); ui.updateSelection(state.selectedIndex, false); });
    
    ui.init({ onProcessStarted: input.resetInactivityTimer });
    ipc.init({ onProcessEnded: input.resetInactivityTimer });
    input.initializeInput();
}

async function initialize() {
    try {
        const loadingOverlay = document.getElementById('loading-overlay');
        const minimumLoadTime = new Promise(resolve => setTimeout(resolve, 2000));
        const mainTasks = appLogic();
        
        await Promise.all([mainTasks, minimumLoadTime]);

        loadingOverlay.classList.add('hidden');
        setTimeout(() => {
            loadingOverlay.style.display = 'none';
        }, 500);
        
        window.addEventListener('beforeunload', () => {
            console.log("Renderer: 'beforeunload' event triggered. Saving final settings.");
            ipc.saveSettingsOnExit(state.appSettings);
        });

    } catch (error) {
        console.error("Renderer: A fatal error occurred during initialization:", error);
        document.body.innerHTML = `<div style="color: red; font-family: monospace; font-size: 16px; padding: 20px; background-color: #111; height: 100vh;">
            <h1>Application Failed to Load</h1><p>A critical error occurred. Please check the developer console (Ctrl+Shift+I) for more details.</p>
            <pre style="color: #ff8080;">${error.stack}</pre></div>`;
    }
}

initialize();